/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2005 by Myricom, Inc.  All rights reserved.                 *
 *************************************************************************/

#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <stdlib.h>
#include <errno.h>
#include <signal.h>
#include <string.h>

#include <sys/types.h>
#include <sys/wait.h>
#include <sys/poll.h>

#define MAX_LABEL_LEN 256

char label[MAX_LABEL_LEN];


void do_io(int rfd, FILE *wfd, int*newline, int *eof, int *written)
{
  char rbuf[1024];
  char obuf[1024+MAX_LABEL_LEN];
  int n;
  int err;
  n = read(rfd,rbuf,sizeof(rbuf));
  err = errno;
  if (n > 0) {
    /* copy read buffer,  inserts label after every new line */
    int i,o;
    for (i =0,o=0;i<n;i++,o++) {
      if (o >= sizeof(obuf) - MAX_LABEL_LEN - 1) {
	fwrite(obuf,1,o,wfd);
	*written += o;
	o = 0;
      }
      if (*newline) {
	if (*written + o > BUFSIZ / 2 || *written + o > 1000) {
	  fwrite(obuf,1,o,wfd);
	  *written += o;
	  o = 0;
	  *written = 0;
	  fflush(stderr);
	  fflush(wfd);
	}
	o += sprintf(obuf+o,"%s",label);
	*newline = 0;
      }
      obuf[o] = rbuf[i];
      if (rbuf[i] == '\n') {
	*newline = 1;
      }
    }
    fwrite(obuf,1,o,wfd);
    *written += o;
  }
  *eof = (n == 0) || (n == -1 && err != EAGAIN);
}

int main(int argc,char *argv[])
{

  pid_t child;
  int status;
  int pipe_stdout[2];
  int pipe_stderr[2];
  int cc;
  struct pollfd ufds[4];
  int stdout_nl = 1;
  int stderr_nl = 1;
  int stdout_eof = 0;
  int stderr_eof = 0;
  int stdout_count = 0;
  int stderr_count = 0;
  const char *s;
  setvbuf(stdout,NULL,_IOFBF,BUFSIZ);
  if (((s = getenv("MXSMPI_LABEL")))
      || ((s = getenv("GMPI_ID")))
      || ((s = getenv("MXMPI_ID")))) {
    sprintf(label,"%s: ",s);
  }
  signal(SIGPIPE, SIG_IGN);
  /* signal(SIGCHLD, handler); */
  if (pipe(pipe_stdout) != 0 || pipe(pipe_stderr) != 0) {
    perror("pipe");
    exit(1);
  }
  child = fork();
  if (child == 0) {
    close(pipe_stdout[0]);
    close(pipe_stderr[0]);
    dup2(pipe_stdout[1],1);
    dup2(pipe_stderr[1],2);
    execvp(argv[1],argv+1);
    fprintf(stderr,"execvp(%s):%s\n", argv[1], strerror(errno));
    _exit(2);
  } else if (child < 0) {
    perror("fork");
    _exit(3);
  }
  close(pipe_stdout[1]);
  close(pipe_stderr[1]);
  fcntl(pipe_stdout[0], F_SETFL, fcntl(pipe_stdout[0],F_GETFL) | O_NONBLOCK);
  fcntl(pipe_stderr[0], F_SETFL, fcntl(pipe_stderr[0],F_GETFL) | O_NONBLOCK);
  ufds[2].events = 0;

  /* workaround bug in libc by first doing line-buffered then
     full-buffered, otherwise the libc bug leaves it unbuffered */
  setvbuf(stderr,NULL,_IOLBF,BUFSIZ);
  setvbuf(stderr,NULL,_IOFBF,BUFSIZ);
  do {
    int n = 0;
    int stdout_idx = -1, stderr_idx = -1;
    if (!stdout_eof) {
      ufds[n].fd = pipe_stdout[0];
      ufds[n].events = POLLIN;
      stdout_idx = n;
      n += 1;
    }
    if (!stderr_eof) {
      ufds[n].fd = pipe_stderr[0];
      ufds[n].events = POLLIN;
      stderr_idx = n;
      n += 1;
    }
    ufds[n].fd = 1;
    ufds[n].events = 0;
    n += 1;
    ufds[n].fd = 2;
    ufds[n].events = 0;
    n += 1;
    cc = poll(ufds,n,300);
    if (cc < 0 && errno == EINTR) {
      continue;
    } else if (cc < 0) {
      perror("poll");
      kill(child,SIGTERM);
      exit(1);
    }
    if (!stdout_eof && ufds[stdout_idx].revents) {
      do_io(pipe_stdout[0],stdout,&stdout_nl, &stdout_eof, &stdout_count);
    }
    if (!stderr_eof && ufds[stderr_idx].revents) {
      do_io(pipe_stderr[0],stderr,&stderr_nl, &stderr_eof, &stderr_count);
    }
    if (ufds[n - 2].revents) {
      break;
    }
    if (ufds[n - 1].revents) {
      break;
    }
    if (cc == 0 && stdout_nl) {
      fflush(stderr);
      fflush(stdout);
      stdout_count = 0;
    }
    if (cc == 0 && stderr_nl) {
      fflush(stderr);
      stderr_count = 0;
    }
  } while (stdout_eof + stderr_eof < 2);
  fflush(stderr);
  fflush(stdout);
  kill(child,SIGTERM);
  waitpid(child,&status,0);
  if (WIFEXITED(status)) {
    if (WEXITSTATUS(status)) {
      fprintf(stderr,"%s%s:exit(%d)\n",label, argv[1], WEXITSTATUS(status));
      fflush(stderr);
    }
    return WEXITSTATUS(status);
  } else {
    fprintf(stderr,"%s%s:signal %d %s\n", label, argv[1], 
	    WTERMSIG(status), WCOREDUMP(status) ? "(core dumped)" : "");
    fflush(stderr);
    raise(WTERMSIG(status));
  }
  _exit(4);
}
